// controllers/authController.js
import User from '../models/User.js';
import jwt from 'jsonwebtoken';
import Joi from 'joi';

const registerUserSchema = Joi.object({
  firstname: Joi.string().min(2).max(50).required(),
  lastname: Joi.string().min(2).max(50).required(),
  email: Joi.string().email().required(),
  phone: Joi.string().pattern(/^[0-9+\-]{10,15}$/).required(),
  leverage: Joi.string().valid("1:100", "1:200", "1:300", "1:400").required(),
  deposit: Joi.string().valid("5000USD", "1000USD", "5500USD", "7500USD").required(),
  isActivated: Joi.boolean().optional(),
  password: Joi.string().min(6).required()
});

// Register user
export const registerUser = async (req, res) => {
  const { error } = registerUserSchema.validate(req.body);
  if (error) {
    return res.status(400).json({ error: error.details[0].message });
  }

  try {
    const existingUser = await User.findOne({ email: req.body.email });
    if (existingUser) {
      return res.status(409).json({ error: "User with this email already exists" });
    }

    const user = await User.create(req.body);
    console.log(' Registered user:', user);
    res.status(201).json({ message: "Registered. Wait for admin activation." });
  } catch (err) {
    console.error(" Registration error:", err.message);
    res.status(500).json({ error: "Registration failed. Try again later." });
  }
};

// Login user
export const loginUser = async (req, res) => {
  const { email, password } = req.body;
  console.log(email, password);

  if (!email || !password) {
    return res.status(400).json({ error: "Email and password are required" });
  }

  try {
    const user = await User.findOne({ email, password });

    if (!user) {
      return res.status(401).json({ message: "Invalid credentials" });
    }

    if (!user.isActive) {
      return res.status(403).json({ message: "Account not activated by admin yet" });
    }

    const token = jwt.sign({ id: user._id, email: user.email }, process.env.JWT_SECRET, {
      expiresIn: '1d'
    });

    res.json({ token });
  } catch (err) {
    console.error(" Login error:", err.message);
    res.status(500).json({ message: "Login failed. Try again later." });
  }
};
