import { FixParser } from '../services/fixParser.js';
import { FixGenerator } from '../services/fixGenerator.js';
import { broadcastMarketData } from '../services/websocketService.js';

const orders = new Map();
let executionCounter = 1;

export class TradingController {
  static handleNewOrderSingle(fixMessage, clientId) {
    const fields = FixParser.parse(fixMessage);
    const clOrdId = fields['11'];
    const symbol = fields['55'];
    const side = fields['54'];
    const orderQty = fields['38'];
    const ordType = fields['40'];
    const price = fields['44'];

    if (!symbol || !side || !orderQty || !ordType) {
      return FixGenerator.generate('8', 
        `11=${clOrdId}|39=8|150=8|58=Missing required fields|`
      );
    }

    const orderId = `ORD${Date.now()}`;
    orders.set(clOrdId, {
      orderId, symbol, side, orderQty, ordType, price,
      status: '0', cumQty: '0', leavesQty: orderQty, clientId
    });

    // Simulate execution after 1 second
    setTimeout(() => this.simulateExecution(clOrdId), 1000);

    return FixGenerator.generate('8', 
      `11=${clOrdId}|37=${orderId}|38=${orderQty}|39=0|` +
      `40=${ordType}|44=${price || ''}|54=${side}|55=${symbol}|` +
      `150=0|151=${orderQty}|`
    );
  }

  static simulateExecution(clOrdId) {
    const order = orders.get(clOrdId);
    if (!order) return;

    const fillQty = Math.min(parseInt(order.orderQty), parseInt(order.orderQty) * 0.7);
    const execId = `EXEC${executionCounter++}`;
    
    order.cumQty = fillQty.toString();
    order.leavesQty = (parseInt(order.orderQty) - fillQty).toString();
    order.status = '1';
    
    const executionReport = FixGenerator.generate('8', 
      `11=${order.clOrdId}|37=${order.orderId}|17=${execId}|` +
      `38=${order.orderQty}|39=1|40=${order.ordType}|44=${order.price || ''}|` +
      `54=${order.side}|55=${order.symbol}|150=F|151=${order.leavesQty}|` +
      `14=${order.cumQty}|32=${fillQty}|31=${order.price || '0'}|`
    );

    // Send via WebSocket
    const wsMessage = executionReport.replace(/\x01/g, '|');
    const client = [...clients].find(c => c.id === order.clientId);
    if (client && client.readyState === client.OPEN) {
      client.send(wsMessage);
    }

    // Complete fill after another second
    if (parseInt(order.leavesQty) > 0) {
      setTimeout(() => this.completeExecution(clOrdId), 1000);
    }
  }

  static completeExecution(clOrdId) {
    const order = orders.get(clOrdId);
    if (!order || order.status === '2') return;

    const execId = `EXEC${executionCounter++}`;
    order.cumQty = order.orderQty;
    order.leavesQty = '0';
    order.status = '2';
    
    const executionReport = FixGenerator.generate('8', 
      `11=${order.clOrdId}|37=${order.orderId}|17=${execId}|` +
      `38=${order.orderQty}|39=2|40=${order.ordType}|44=${order.price || ''}|` +
      `54=${order.side}|55=${order.symbol}|150=F|151=0|` +
      `14=${order.cumQty}|32=${order.leavesQty}|31=${order.price || '0'}|`
    );

    const wsMessage = executionReport.replace(/\x01/g, '|');
    const client = [...clients].find(c => c.id === order.clientId);
    if (client && client.readyState === client.OPEN) {
      client.send(wsMessage);
    }
  }

  static handleOrderCancelRequest(fixMessage) {
    const fields = FixParser.parse(fixMessage);
    const clOrdId = fields['11'];
    const order = orders.get(clOrdId);
    
    if (!order) {
      return FixGenerator.generate('8', 
        `11=${clOrdId}|39=8|150=8|58=Order not found|`
      );
    }

    order.status = '4';
    order.leavesQty = '0';

    return FixGenerator.generate('8', 
      `11=${clOrdId}|37=${order.orderId}|38=${order.orderQty}|39=4|` +
      `40=${order.ordType}|44=${order.price || ''}|54=${order.side}|` +
      `55=${order.symbol}|150=4|151=0|`
    );
  }
}